/*
 * Decompiled with CFR 0.152.
 */
package net.minecraft.world.level.chunk.storage;

import com.google.common.annotations.VisibleForTesting;
import com.mojang.logging.LogUtils;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.IntBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import javax.annotation.Nullable;
import net.minecraft.Util;
import net.minecraft.world.level.ChunkPos;
import net.minecraft.world.level.chunk.storage.RegionBitmap;
import net.minecraft.world.level.chunk.storage.RegionFileVersion;
import org.slf4j.Logger;

public class RegionFile
implements AutoCloseable {
    private static final Logger LOGGER = LogUtils.getLogger();
    private static final int SECTOR_BYTES = 4096;
    @VisibleForTesting
    protected static final int SECTOR_INTS = 1024;
    private static final int CHUNK_HEADER_SIZE = 5;
    private static final int HEADER_OFFSET = 0;
    private static final ByteBuffer PADDING_BUFFER = ByteBuffer.allocateDirect(1);
    private static final String EXTERNAL_FILE_EXTENSION = ".mcc";
    private static final int EXTERNAL_STREAM_FLAG = 128;
    private static final int EXTERNAL_CHUNK_THRESHOLD = 256;
    private static final int CHUNK_NOT_PRESENT = 0;
    private final FileChannel file;
    private final Path externalFileDir;
    final RegionFileVersion version;
    private final ByteBuffer header = ByteBuffer.allocateDirect(8192);
    private final IntBuffer offsets;
    private final IntBuffer timestamps;
    @VisibleForTesting
    protected final RegionBitmap usedSectors = new RegionBitmap();

    public RegionFile(Path p_196950_, Path p_196951_, boolean p_196952_) throws IOException {
        this(p_196950_, p_196951_, RegionFileVersion.VERSION_DEFLATE, p_196952_);
    }

    public RegionFile(Path pRegionFile, Path pContainingFolder, RegionFileVersion pVersion, boolean pSync) throws IOException {
        this.version = pVersion;
        if (!Files.isDirectory(pContainingFolder, new LinkOption[0])) {
            throw new IllegalArgumentException("Expected directory, got " + pContainingFolder.toAbsolutePath());
        }
        this.externalFileDir = pContainingFolder;
        this.offsets = this.header.asIntBuffer();
        this.offsets.limit(1024);
        this.header.position(4096);
        this.timestamps = this.header.asIntBuffer();
        this.file = pSync ? FileChannel.open(pRegionFile, StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE, StandardOpenOption.DSYNC) : FileChannel.open(pRegionFile, StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
        this.usedSectors.force(0, 2);
        this.header.position(0);
        int i = this.file.read(this.header, 0L);
        if (i != -1) {
            if (i != 8192) {
                LOGGER.warn("Region file {} has truncated header: {}", (Object)pRegionFile, (Object)i);
            }
            long j = Files.size(pRegionFile);
            int k = 0;
            while (k < 1024) {
                int l = this.offsets.get(k);
                if (l != 0) {
                    int i1 = RegionFile.getSectorNumber(l);
                    int j1 = RegionFile.getNumSectors(l);
                    if (i1 < 2) {
                        LOGGER.warn("Region file {} has invalid sector at index: {}; sector {} overlaps with header", new Object[]{pRegionFile, k, i1});
                        this.offsets.put(k, 0);
                    } else if (j1 == 0) {
                        LOGGER.warn("Region file {} has an invalid sector at index: {}; size has to be > 0", (Object)pRegionFile, (Object)k);
                        this.offsets.put(k, 0);
                    } else if ((long)i1 * 4096L > j) {
                        LOGGER.warn("Region file {} has an invalid sector at index: {}; sector {} is out of bounds", new Object[]{pRegionFile, k, i1});
                        this.offsets.put(k, 0);
                    } else {
                        this.usedSectors.force(i1, j1);
                    }
                }
                ++k;
            }
        }
    }

    private Path getExternalChunkPath(ChunkPos pChunkPos) {
        String s = "c." + pChunkPos.x + "." + pChunkPos.z + EXTERNAL_FILE_EXTENSION;
        return this.externalFileDir.resolve(s);
    }

    @Nullable
    public synchronized DataInputStream getChunkDataInputStream(ChunkPos pChunkPos) throws IOException {
        int i = this.getOffset(pChunkPos);
        if (i == 0) {
            return null;
        }
        int j = RegionFile.getSectorNumber(i);
        int k = RegionFile.getNumSectors(i);
        int l = k * 4096;
        ByteBuffer bytebuffer = ByteBuffer.allocate(l);
        this.file.read(bytebuffer, j * 4096);
        bytebuffer.flip();
        if (bytebuffer.remaining() < 5) {
            LOGGER.error("Chunk {} header is truncated: expected {} but read {}", new Object[]{pChunkPos, l, bytebuffer.remaining()});
            return null;
        }
        int i1 = bytebuffer.getInt();
        byte b0 = bytebuffer.get();
        if (i1 == 0) {
            LOGGER.warn("Chunk {} is allocated, but stream is missing", (Object)pChunkPos);
            return null;
        }
        int j1 = i1 - 1;
        if (RegionFile.isExternalStreamChunk(b0)) {
            if (j1 != 0) {
                LOGGER.warn("Chunk has both internal and external streams");
            }
            return this.createExternalChunkInputStream(pChunkPos, RegionFile.getExternalChunkVersion(b0));
        }
        if (j1 > bytebuffer.remaining()) {
            LOGGER.error("Chunk {} stream is truncated: expected {} but read {}", new Object[]{pChunkPos, j1, bytebuffer.remaining()});
            return null;
        }
        if (j1 < 0) {
            LOGGER.error("Declared size {} of chunk {} is negative", (Object)i1, (Object)pChunkPos);
            return null;
        }
        return this.createChunkInputStream(pChunkPos, b0, RegionFile.createStream(bytebuffer, j1));
    }

    private static int getTimestamp() {
        return (int)(Util.getEpochMillis() / 1000L);
    }

    private static boolean isExternalStreamChunk(byte pVersionByte) {
        return (pVersionByte & 0x80) != 0;
    }

    private static byte getExternalChunkVersion(byte pVersionByte) {
        return (byte)(pVersionByte & 0xFFFFFF7F);
    }

    @Nullable
    private DataInputStream createChunkInputStream(ChunkPos pChunkPos, byte pVersionByte, InputStream pInputStream) throws IOException {
        RegionFileVersion regionfileversion = RegionFileVersion.fromId(pVersionByte);
        if (regionfileversion == null) {
            LOGGER.error("Chunk {} has invalid chunk stream version {}", (Object)pChunkPos, (Object)pVersionByte);
            return null;
        }
        return new DataInputStream(regionfileversion.wrap(pInputStream));
    }

    @Nullable
    private DataInputStream createExternalChunkInputStream(ChunkPos pChunkPos, byte pVersionByte) throws IOException {
        Path path = this.getExternalChunkPath(pChunkPos);
        if (!Files.isRegularFile(path, new LinkOption[0])) {
            LOGGER.error("External chunk path {} is not file", (Object)path);
            return null;
        }
        return this.createChunkInputStream(pChunkPos, pVersionByte, Files.newInputStream(path, new OpenOption[0]));
    }

    private static ByteArrayInputStream createStream(ByteBuffer pSourceBuffer, int pLength) {
        return new ByteArrayInputStream(pSourceBuffer.array(), pSourceBuffer.position(), pLength);
    }

    private int packSectorOffset(int pSectorOffset, int pSectorCount) {
        return pSectorOffset << 8 | pSectorCount;
    }

    private static int getNumSectors(int pPackedSectorOffset) {
        return pPackedSectorOffset & 0xFF;
    }

    private static int getSectorNumber(int pPackedSectorOffset) {
        return pPackedSectorOffset >> 8 & 0xFFFFFF;
    }

    private static int sizeToSectors(int pSize) {
        return (pSize + 4096 - 1) / 4096;
    }

    public boolean doesChunkExist(ChunkPos pChunkPos) {
        block12: {
            int l;
            int k;
            block14: {
                block13: {
                    byte b0;
                    block10: {
                        block11: {
                            ByteBuffer bytebuffer;
                            block9: {
                                int i = this.getOffset(pChunkPos);
                                if (i == 0) {
                                    return false;
                                }
                                int j = RegionFile.getSectorNumber(i);
                                k = RegionFile.getNumSectors(i);
                                bytebuffer = ByteBuffer.allocate(5);
                                try {
                                    this.file.read(bytebuffer, j * 4096);
                                    bytebuffer.flip();
                                    if (bytebuffer.remaining() == 5) break block9;
                                    return false;
                                }
                                catch (IOException ioexception) {
                                    return false;
                                }
                            }
                            l = bytebuffer.getInt();
                            b0 = bytebuffer.get();
                            if (!RegionFile.isExternalStreamChunk(b0)) break block10;
                            if (RegionFileVersion.isValidVersion(RegionFile.getExternalChunkVersion(b0))) break block11;
                            return false;
                        }
                        if (!Files.isRegularFile(this.getExternalChunkPath(pChunkPos), new LinkOption[0])) {
                            return false;
                        }
                        break block12;
                    }
                    if (RegionFileVersion.isValidVersion(b0)) break block13;
                    return false;
                }
                if (l != 0) break block14;
                return false;
            }
            int i1 = l - 1;
            if (i1 >= 0 && i1 <= 4096 * k) break block12;
            return false;
        }
        return true;
    }

    public DataOutputStream getChunkDataOutputStream(ChunkPos pChunkPos) throws IOException {
        return new DataOutputStream(this.version.wrap(new ChunkBuffer(pChunkPos)));
    }

    public void flush() throws IOException {
        this.file.force(true);
    }

    public void clear(ChunkPos pChunkPos) throws IOException {
        int i = RegionFile.getOffsetIndex(pChunkPos);
        int j = this.offsets.get(i);
        if (j != 0) {
            this.offsets.put(i, 0);
            this.timestamps.put(i, RegionFile.getTimestamp());
            this.writeHeader();
            Files.deleteIfExists(this.getExternalChunkPath(pChunkPos));
            this.usedSectors.free(RegionFile.getSectorNumber(j), RegionFile.getNumSectors(j));
        }
    }

    protected synchronized void write(ChunkPos pChunkPos, ByteBuffer pChunkData) throws IOException {
        CommitOp regionfile$commitop;
        int k1;
        int i = RegionFile.getOffsetIndex(pChunkPos);
        int j = this.offsets.get(i);
        int k = RegionFile.getSectorNumber(j);
        int l = RegionFile.getNumSectors(j);
        int i1 = pChunkData.remaining();
        int j1 = RegionFile.sizeToSectors(i1);
        if (j1 >= 256) {
            Path path = this.getExternalChunkPath(pChunkPos);
            LOGGER.warn("Saving oversized chunk {} ({} bytes} to external file {}", new Object[]{pChunkPos, i1, path});
            j1 = 1;
            k1 = this.usedSectors.allocate(j1);
            regionfile$commitop = this.writeToExternalFile(path, pChunkData);
            ByteBuffer bytebuffer = this.createExternalStub();
            this.file.write(bytebuffer, k1 * 4096);
        } else {
            k1 = this.usedSectors.allocate(j1);
            regionfile$commitop = () -> Files.deleteIfExists(this.getExternalChunkPath(pChunkPos));
            this.file.write(pChunkData, k1 * 4096);
        }
        this.offsets.put(i, this.packSectorOffset(k1, j1));
        this.timestamps.put(i, RegionFile.getTimestamp());
        this.writeHeader();
        regionfile$commitop.run();
        if (k != 0) {
            this.usedSectors.free(k, l);
        }
    }

    private ByteBuffer createExternalStub() {
        ByteBuffer bytebuffer = ByteBuffer.allocate(5);
        bytebuffer.putInt(1);
        bytebuffer.put((byte)(this.version.getId() | 0x80));
        bytebuffer.flip();
        return bytebuffer;
    }

    private CommitOp writeToExternalFile(Path pExternalChunkFile, ByteBuffer pChunkData) throws IOException {
        Path path = Files.createTempFile(this.externalFileDir, "tmp", null, new FileAttribute[0]);
        try (FileChannel filechannel = FileChannel.open(path, StandardOpenOption.CREATE, StandardOpenOption.WRITE);){
            pChunkData.position(5);
            filechannel.write(pChunkData);
        }
        return () -> Files.move(path, pExternalChunkFile, StandardCopyOption.REPLACE_EXISTING);
    }

    private void writeHeader() throws IOException {
        this.header.position(0);
        this.file.write(this.header, 0L);
    }

    private int getOffset(ChunkPos pChunkPos) {
        return this.offsets.get(RegionFile.getOffsetIndex(pChunkPos));
    }

    public boolean hasChunk(ChunkPos pChunkPos) {
        return this.getOffset(pChunkPos) != 0;
    }

    private static int getOffsetIndex(ChunkPos pChunkPos) {
        return pChunkPos.getRegionLocalX() + pChunkPos.getRegionLocalZ() * 32;
    }

    @Override
    public void close() throws IOException {
        try {
            this.padToFullSector();
        }
        finally {
            try {
                this.file.force(true);
            }
            finally {
                this.file.close();
            }
        }
    }

    private void padToFullSector() throws IOException {
        int j;
        int i = (int)this.file.size();
        if (i != (j = RegionFile.sizeToSectors(i) * 4096)) {
            ByteBuffer bytebuffer = PADDING_BUFFER.duplicate();
            bytebuffer.position(0);
            this.file.write(bytebuffer, j - 1);
        }
    }

    class ChunkBuffer
    extends ByteArrayOutputStream {
        private final ChunkPos pos;

        public ChunkBuffer(ChunkPos p_63696_) {
            super(8096);
            super.write(0);
            super.write(0);
            super.write(0);
            super.write(0);
            super.write(RegionFile.this.version.getId());
            this.pos = p_63696_;
        }

        @Override
        public void close() throws IOException {
            ByteBuffer bytebuffer = ByteBuffer.wrap(this.buf, 0, this.count);
            bytebuffer.putInt(0, this.count - 5 + 1);
            RegionFile.this.write(this.pos, bytebuffer);
        }
    }

    static interface CommitOp {
        public void run() throws IOException;
    }
}

